#!/usr/bin/env python3
"""
Example usage of Qdrant Hybrid Search.

Before running:
1. Start Qdrant: docker-compose up -d
2. Set up .env with SEARCH_EMBEDDING_API_KEY
3. Run indexing example first, then search examples
"""

import os
from pathlib import Path
from typing import TypedDict

from dotenv import load_dotenv
load_dotenv()


# =============================================================================
# Example 1: Indexing API endpoints
# =============================================================================

def example_index_api_endpoints():
    """Index API endpoints from JSONL file."""
    from qdrant_hybrid_search import SearchConfig, IndexConfig, QdrantIndexer

    # Check for API key
    api_key = os.environ.get("SEARCH_EMBEDDING_API_KEY") or os.environ.get("OPENROUTER_API_KEY")
    if not api_key:
        print("Error: Set SEARCH_EMBEDDING_API_KEY or OPENROUTER_API_KEY")
        return

    # Qdrant configuration
    search_config = SearchConfig(
        qdrant_url="http://localhost:6333",
        collection_name="api_endpoints",
        embedding_api_key=api_key,
    )

    # Index configuration:
    # - text_fields: combined for semantic search + BM25 keyword search
    # - payload_indexes: for filtering
    index_config = IndexConfig(
        id_field="operation_id",
        text_fields=["summary", "path", "description", "method", "tags"],
        payload_indexes={
            "resource_type": "keyword",
            "effect_type": "keyword",
            "method": "keyword",
        },
    )

    indexer = QdrantIndexer(search_config, index_config)

    # Create collection (reset if exists)
    indexer.create_collection(reset=True)
    indexer.create_payload_indexes()

    # Index from JSONL
    jsonl_path = Path(__file__).parent.parent / "api-index" / "api-endpoints.jsonl"
    if jsonl_path.exists():
        indexer.index_from_jsonl(jsonl_path)
        print(f"\n{indexer.get_collection_info()}")
    else:
        print(f"File not found: {jsonl_path}")
        print("Create a sample file or adjust the path")


# =============================================================================
# Example 2: Simple search
# =============================================================================

def example_simple_search():
    """Direct search without LangGraph."""
    from qdrant_hybrid_search import SearchConfig, QdrantHybridSearch

    api_key = os.environ.get("SEARCH_EMBEDDING_API_KEY") or os.environ.get("OPENROUTER_API_KEY")

    config = SearchConfig(
        qdrant_url="http://localhost:6333",
        collection_name="api_endpoints",
        embedding_api_key=api_key,
        top_k=5,
        use_reranking=False,  # Disable for speed
    )

    search = QdrantHybridSearch(config)

    print("\n=== Simple Search ===\n")

    # Basic search
    results = search.search("send document for signature")
    print(f"Query: 'send document for signature'")
    print(f"Found: {len(results)} results\n")
    for r in results:
        print(f"  [{r.score:.3f}] {r.payload.get('operation_id')}")
        print(f"           {r.payload.get('summary')}")


# =============================================================================
# Example 3: Search with filters
# =============================================================================

def example_filtered_search():
    """Search with payload filters."""
    from qdrant_hybrid_search import SearchConfig, QdrantHybridSearch

    api_key = os.environ.get("SEARCH_EMBEDDING_API_KEY") or os.environ.get("OPENROUTER_API_KEY")

    config = SearchConfig(
        qdrant_url="http://localhost:6333",
        collection_name="api_endpoints",
        embedding_api_key=api_key,
        top_k=5,
    )

    search = QdrantHybridSearch(config)

    print("\n=== Filtered Search ===\n")

    # Search only READ operations on documents
    results = search.search(
        "get user documents",
        filters={
            "effect_type": "read",
            "resource_type": "document",
        },
    )

    print(f"Query: 'get user documents' (effect_type=read, resource_type=document)")
    print(f"Found: {len(results)} results\n")
    for r in results:
        print(f"  [{r.score:.3f}] {r.payload.get('method')} {r.payload.get('path')}")


# =============================================================================
# Example 4: LangGraph simple node
# =============================================================================

def example_langgraph_simple():
    """Use as a simple LangGraph node."""
    from langgraph.graph import StateGraph, END
    from qdrant_hybrid_search import create_simple_search_node, SearchConfig

    api_key = os.environ.get("SEARCH_EMBEDDING_API_KEY") or os.environ.get("OPENROUTER_API_KEY")

    class SimpleState(TypedDict):
        query: str
        search_results: list
        search_queries: list

    config = SearchConfig(
        qdrant_url="http://localhost:6333",
        collection_name="api_endpoints",
        embedding_api_key=api_key,
        top_k=5,
    )

    search_node = create_simple_search_node(config)

    graph = StateGraph(SimpleState)
    graph.add_node("search", search_node)
    graph.set_entry_point("search")
    graph.add_edge("search", END)

    app = graph.compile()

    print("\n=== LangGraph Simple Node ===\n")

    result = app.invoke({"query": "resend signing invitation"})
    print(f"Query: 'resend signing invitation'")
    print(f"Found: {len(result['search_results'])} results")
    for r in result["search_results"][:3]:
        print(f"  - {r['payload'].get('operation_id')}: {r['payload'].get('summary')}")


# =============================================================================
# Example 5: LangGraph with LLM query generation
# =============================================================================

def example_langgraph_full():
    """Full LangGraph with LLM-based query generation."""
    from langchain_openai import ChatOpenAI
    from langgraph.graph import StateGraph, END
    from qdrant_hybrid_search import create_search_node, SearchNodeConfig, SearchConfig

    api_key = os.environ.get("SEARCH_EMBEDDING_API_KEY") or os.environ.get("OPENROUTER_API_KEY")
    openai_key = os.environ.get("OPENAI_API_KEY")

    if not openai_key:
        print("Skipping: OPENAI_API_KEY not set")
        return

    class AgentState(TypedDict):
        user_command: str
        context: dict
        search_queries: list
        search_results: list
        next_action: str

    node_config = SearchNodeConfig(
        search_config=SearchConfig(
            qdrant_url="http://localhost:6333",
            collection_name="api_endpoints",
            embedding_api_key=api_key,
            top_k=10,
        ),
        generate_queries=True,
        max_queries=3,
        state_input_key="user_command",
        state_output_key="search_results",
    )

    search_node = create_search_node(node_config)
    llm = ChatOpenAI(model="gpt-4o-mini", temperature=0)

    graph = StateGraph(AgentState)
    graph.add_node("search", lambda state: search_node(state, llm))
    graph.set_entry_point("search")
    graph.add_edge("search", END)

    app = graph.compile()

    print("\n=== LangGraph with LLM Query Generation ===\n")

    result = app.invoke({
        "user_command": "надішли нагадування підписанту про документ",
        "context": {"tenant_id": "test123"},
    })

    print(f"User command: 'надішли нагадування підписанту про документ'")
    print(f"Generated queries: {result['search_queries']}")
    print(f"Found: {len(result['search_results'])} results")


# =============================================================================
# Example 6: Custom synonyms
# =============================================================================

def example_custom_synonyms():
    """Search with custom domain synonyms."""
    from qdrant_hybrid_search import SearchConfig, QdrantHybridSearch, SynonymExpander

    api_key = os.environ.get("SEARCH_EMBEDDING_API_KEY") or os.environ.get("OPENROUTER_API_KEY")

    # Domain-specific synonyms
    synonyms = {
        "reminder": ["resend", "follow up", "notify", "ping"],
        "sign": ["execute", "approve", "finalize", "complete"],
        "document": ["doc", "file", "contract", "agreement"],
        "invite": ["request", "invitation", "signing request"],
    }

    config = SearchConfig(
        qdrant_url="http://localhost:6333",
        collection_name="api_endpoints",
        embedding_api_key=api_key,
        top_k=5,
    )

    search = QdrantHybridSearch(
        config=config,
        synonym_expander=SynonymExpander(synonyms),
    )

    print("\n=== Custom Synonyms ===\n")

    results = search.search("send reminder to signer")
    print(f"Query: 'send reminder to signer' (with custom synonyms)")
    print(f"Found: {len(results)} results\n")
    for r in results:
        print(f"  [{r.score:.3f}] {r.payload.get('operation_id')}")


# =============================================================================
# Main
# =============================================================================

if __name__ == "__main__":
    import sys

    examples = {
        "index": example_index_api_endpoints,
        "search": example_simple_search,
        "filter": example_filtered_search,
        "langgraph": example_langgraph_simple,
        "langgraph-llm": example_langgraph_full,
        "synonyms": example_custom_synonyms,
    }

    if len(sys.argv) > 1:
        name = sys.argv[1]
        if name in examples:
            examples[name]()
        else:
            print(f"Unknown example: {name}")
            print(f"Available: {', '.join(examples.keys())}")
    else:
        print("Qdrant Hybrid Search Examples")
        print("=" * 40)
        print("\nUsage: python example.py <example_name>\n")
        print("Available examples:")
        print("  index        - Index API endpoints from JSONL")
        print("  search       - Simple hybrid search")
        print("  filter       - Search with filters")
        print("  langgraph    - Simple LangGraph node")
        print("  langgraph-llm - LangGraph with LLM query generation")
        print("  synonyms     - Custom domain synonyms")
        print("\nStart with: python example.py index")
