"""Configuration for Qdrant Hybrid Search Service."""

from pydantic_settings import BaseSettings
from pydantic import Field


class SearchConfig(BaseSettings):
    """Configuration for the search service.

    All settings can be overridden via environment variables with SEARCH_ prefix.
    Example: SEARCH_QDRANT_URL=http://localhost:6333
    """

    # Qdrant settings
    qdrant_url: str = Field(
        default="http://localhost:6333",
        description="Qdrant server URL",
    )
    collection_name: str = Field(
        default="search_index",
        description="Name of the Qdrant collection",
    )

    # Embedding settings
    embedding_provider: str = Field(
        default="openrouter",
        description="Embedding provider: 'openrouter' or 'openai'",
    )
    embedding_api_key: str = Field(
        default="",
        description="API key for embedding provider",
    )
    embedding_base_url: str = Field(
        default="https://openrouter.ai/api/v1",
        description="Base URL for embedding API",
    )
    embedding_model: str = Field(
        default="openai/text-embedding-3-small",
        description="Embedding model name",
    )
    embedding_dims: int = Field(
        default=1536,
        description="Embedding dimensions",
    )

    # Search settings
    top_k: int = Field(
        default=10,
        description="Default number of results to return",
    )
    min_score: float = Field(
        default=0.1,
        description="Minimum relevance score threshold",
    )
    use_reranking: bool = Field(
        default=True,
        description="Whether to use LLM-based reranking",
    )
    rerank_model: str = Field(
        default="anthropic/claude-3-haiku",
        description="Model for reranking",
    )

    # Matryoshka dimensions for progressive search
    matryoshka_dims: list[int] = Field(
        default=[64, 128, 256, 512],
        description="Dimensions for matryoshka embeddings",
    )

    # RRF fusion parameter
    rrf_k: int = Field(
        default=60,
        description="K parameter for Reciprocal Rank Fusion",
    )

    model_config = {
        "env_prefix": "SEARCH_",
        "env_file": ".env",
        "extra": "ignore",
    }


class IndexConfig(BaseSettings):
    """Configuration for indexing documents.

    Defines which fields to use for text search (BM25/sparse vectors)
    and which fields to create payload indexes for filtering.
    """

    # Fields for text search (combined for dense embedding + BM25 sparse vector)
    text_fields: list[str] = Field(
        default=["content"],
        description="Fields to combine for text search (dense + sparse/BM25 vectors)",
    )

    # Primary ID field in source documents
    id_field: str = Field(
        default="id",
        description="Field name for document ID",
    )

    # Payload indexes for filtering (field_name -> type)
    # Types: keyword, integer, float, bool, text
    payload_indexes: dict[str, str] = Field(
        default={},
        description="Payload fields to index for filtering. Format: {field: type}",
    )

    # Whether to store original text in payload
    store_text: bool = Field(
        default=True,
        description="Store combined search text in payload",
    )

    # Batch size for indexing
    batch_size: int = Field(
        default=50,
        description="Number of documents per batch",
    )

    model_config = {
        "env_prefix": "INDEX_",
        "extra": "ignore",
    }
