"""Embedding client for vector generation."""

import httpx
from typing import Protocol

from .config import SearchConfig


class EmbeddingClient(Protocol):
    """Protocol for embedding clients."""

    def get_embedding(self, text: str) -> list[float]:
        """Get embedding for a single text."""
        ...

    def get_embeddings_batch(self, texts: list[str]) -> list[list[float]]:
        """Get embeddings for multiple texts."""
        ...


class OpenRouterEmbeddingClient:
    """Embedding client using OpenRouter API."""

    def __init__(self, config: SearchConfig):
        self.config = config
        self._cache: dict[str, list[float]] = {}

    def get_embedding(self, text: str) -> list[float]:
        """Get embedding for a single text with caching."""
        if text in self._cache:
            return self._cache[text]

        if not self.config.embedding_api_key:
            raise ValueError("embedding_api_key not set in config")

        response = httpx.post(
            f"{self.config.embedding_base_url}/embeddings",
            headers={
                "Authorization": f"Bearer {self.config.embedding_api_key}",
                "Content-Type": "application/json",
            },
            json={
                "model": self.config.embedding_model,
                "input": text,
                "dimensions": self.config.embedding_dims,
            },
            timeout=30.0,
        )
        response.raise_for_status()
        data = response.json()
        embedding = data["data"][0]["embedding"]

        self._cache[text] = embedding
        return embedding

    def get_embeddings_batch(self, texts: list[str]) -> list[list[float]]:
        """Get embeddings for multiple texts."""
        if not self.config.embedding_api_key:
            raise ValueError("embedding_api_key not set in config")

        response = httpx.post(
            f"{self.config.embedding_base_url}/embeddings",
            headers={
                "Authorization": f"Bearer {self.config.embedding_api_key}",
                "Content-Type": "application/json",
            },
            json={
                "model": self.config.embedding_model,
                "input": texts,
                "dimensions": self.config.embedding_dims,
            },
            timeout=120.0,
        )
        response.raise_for_status()
        data = response.json()
        return [item["embedding"] for item in data["data"]]

    def clear_cache(self) -> None:
        """Clear the embedding cache."""
        self._cache.clear()


def create_embedding_client(config: SearchConfig) -> EmbeddingClient:
    """Factory function to create embedding client based on config."""
    if config.embedding_provider == "openrouter":
        return OpenRouterEmbeddingClient(config)
    elif config.embedding_provider == "openai":
        # OpenAI uses the same API format
        return OpenRouterEmbeddingClient(config)
    else:
        raise ValueError(f"Unknown embedding provider: {config.embedding_provider}")
