"""LangGraph node for integrating Qdrant hybrid search."""

from dataclasses import dataclass, field
from typing import Any, Callable, TypedDict

from langchain_core.messages import HumanMessage, SystemMessage
from langchain_core.language_models import BaseChatModel

from .config import SearchConfig
from .models import SearchResult, SearchQuery
from .search import QdrantHybridSearch


@dataclass
class SearchNodeConfig:
    """Configuration for the LangGraph search node.

    Attributes:
        search_config: Qdrant search configuration.
        generate_queries: Whether to use LLM to generate search queries.
        max_queries: Maximum number of queries to generate.
        state_input_key: Key in state dict for user input/command.
        state_output_key: Key in state dict for search results.
        query_generation_prompt: System prompt for query generation.
    """

    search_config: SearchConfig = field(default_factory=SearchConfig)
    generate_queries: bool = True
    max_queries: int = 5
    state_input_key: str = "user_command"
    state_output_key: str = "search_results"
    query_generation_prompt: str = """Analyze the user's request and generate search queries.
Generate 1-5 targeted search queries to find relevant items.
Focus on:
1. Main action or intent
2. Resources or entities involved
3. Any specific attributes mentioned

Return queries as a list, one per line."""


class SearchNodeOutput(TypedDict, total=False):
    """Output type for the search node."""

    search_queries: list[str]
    search_results: list[dict[str, Any]]
    current_node: str
    next_action: str


class SearchQueryGenerator:
    """Generates search queries from user input using LLM."""

    def __init__(self, config: SearchNodeConfig):
        self.config = config

    def generate(
        self,
        user_input: str,
        llm: BaseChatModel,
        context: dict[str, Any] | None = None,
    ) -> list[str]:
        """Generate search queries from user input.

        Args:
            user_input: The user's command or request.
            llm: Language model for query generation.
            context: Optional context dict with additional info.

        Returns:
            List of search queries.
        """
        context = context or {}

        prompt = f"""{self.config.query_generation_prompt}

User Input: {user_input}

Context:
{self._format_context(context)}

Generate search queries (one per line):"""

        try:
            response = llm.invoke([
                SystemMessage(content="You are a search query generator."),
                HumanMessage(content=prompt),
            ])

            content = response.content if hasattr(response, "content") else str(response)
            queries = [
                q.strip().lstrip("0123456789.-) ")
                for q in content.strip().split("\n")
                if q.strip()
            ]
            return queries[:self.config.max_queries]

        except Exception:
            # Fallback to using user input as query
            return [user_input]

    def _format_context(self, context: dict[str, Any]) -> str:
        """Format context dict for prompt."""
        if not context:
            return "None"
        return "\n".join(f"- {k}: {v}" for k, v in context.items())


def create_search_node(
    config: SearchNodeConfig | None = None,
    search: QdrantHybridSearch | None = None,
) -> Callable[[dict[str, Any], BaseChatModel], SearchNodeOutput]:
    """Create a LangGraph-compatible search node function.

    Args:
        config: Search node configuration.
        search: Pre-configured search instance.

    Returns:
        A function that can be used as a LangGraph node.

    Example:
        ```python
        from langgraph.graph import StateGraph
        from src import create_search_node, SearchNodeConfig

        config = SearchNodeConfig(
            state_input_key="user_input",
            state_output_key="results",
        )

        search_node = create_search_node(config)

        graph = StateGraph(MyState)
        graph.add_node("search", lambda state: search_node(state, llm))
        ```
    """
    config = config or SearchNodeConfig()
    search_instance = search or QdrantHybridSearch(config.search_config)
    query_generator = SearchQueryGenerator(config)

    def search_node(
        state: dict[str, Any],
        llm: BaseChatModel,
    ) -> SearchNodeOutput:
        """LangGraph search node implementation.

        Args:
            state: Current graph state.
            llm: Language model for query generation.

        Returns:
            Dict with search results and metadata.
        """
        user_input = state.get(config.state_input_key, "")

        if not user_input:
            return {
                "search_queries": [],
                config.state_output_key: [],
                "current_node": "search",
                "next_action": "error",
            }

        # Generate or use direct queries
        if config.generate_queries:
            context = {
                k: v for k, v in state.items()
                if k not in (config.state_input_key, config.state_output_key)
                and isinstance(v, (str, int, float, bool))
            }
            queries = query_generator.generate(user_input, llm, context)
        else:
            queries = [user_input]

        # Execute searches
        all_results: list[SearchResult] = []
        seen_ids: set[str] = set()

        for query in queries:
            if not query:
                continue

            results = search_instance.search(query)

            for r in results:
                if r.id not in seen_ids:
                    seen_ids.add(r.id)
                    all_results.append(r)

        # Sort by score and limit
        all_results.sort(key=lambda x: x.score, reverse=True)
        all_results = all_results[:config.search_config.top_k * 2]

        return {
            "search_queries": queries,
            config.state_output_key: [r.to_dict() for r in all_results],
            "current_node": "search",
            "next_action": "continue",
        }

    return search_node


def create_simple_search_node(
    config: SearchConfig | None = None,
) -> Callable[[dict[str, Any]], SearchNodeOutput]:
    """Create a simple search node without LLM query generation.

    This is a simpler version that takes the user input directly as query.

    Args:
        config: Search configuration.

    Returns:
        A function that can be used as a LangGraph node.

    Example:
        ```python
        from langgraph.graph import StateGraph
        from src import create_simple_search_node

        search_node = create_simple_search_node()

        graph = StateGraph(MyState)
        graph.add_node("search", search_node)
        ```
    """
    config = config or SearchConfig()
    search = QdrantHybridSearch(config)

    def search_node(state: dict[str, Any]) -> SearchNodeOutput:
        """Simple search node implementation."""
        query = state.get("query") or state.get("user_input") or state.get("user_command", "")

        if not query:
            return {
                "search_queries": [],
                "search_results": [],
                "current_node": "search",
                "next_action": "error",
            }

        results = search.search(query)

        return {
            "search_queries": [query],
            "search_results": [r.to_dict() for r in results],
            "current_node": "search",
            "next_action": "continue",
        }

    return search_node
