"""Data models for Qdrant Hybrid Search Service."""

from dataclasses import dataclass, field
from typing import Any


@dataclass
class SearchResult:
    """Single search result from Qdrant."""

    id: str
    score: float
    payload: dict[str, Any] = field(default_factory=dict)

    # Common payload fields (optional, for convenience)
    @property
    def title(self) -> str:
        return self.payload.get("title", "")

    @property
    def description(self) -> str:
        return self.payload.get("description", "")

    @property
    def content(self) -> str:
        return self.payload.get("content", "")

    def get(self, key: str, default: Any = None) -> Any:
        """Get a payload field by key."""
        return self.payload.get(key, default)

    def to_dict(self) -> dict[str, Any]:
        """Convert to dictionary."""
        return {
            "id": self.id,
            "score": self.score,
            "payload": self.payload,
        }

    @classmethod
    def from_qdrant_point(cls, point: Any, score: float) -> "SearchResult":
        """Create from Qdrant ScoredPoint."""
        return cls(
            id=str(point.id) if hasattr(point, "id") else str(point.payload.get("id", "")),
            score=score,
            payload=point.payload or {},
        )


@dataclass
class SearchQuery:
    """Search query with optional filters."""

    query: str
    top_k: int = 10
    min_score: float = 0.0
    filters: dict[str, Any] = field(default_factory=dict)
    use_reranking: bool = True

    # Optional metadata
    metadata: dict[str, Any] = field(default_factory=dict)


@dataclass
class IndexDocument:
    """Document to be indexed in Qdrant."""

    id: str | int
    content: str
    payload: dict[str, Any] = field(default_factory=dict)

    def get_search_text(self) -> str:
        """Get text for embedding generation."""
        parts = [self.content]

        # Add common payload fields
        for key in ["title", "summary", "description", "tags"]:
            value = self.payload.get(key)
            if value:
                if isinstance(value, list):
                    parts.extend(value)
                else:
                    parts.append(str(value))

        return " ".join(filter(None, parts))
