"""Synonym expansion for search queries."""

from typing import Callable


# Default synonyms dictionary - can be customized
DEFAULT_SYNONYMS: dict[str, list[str]] = {
    # Send/Invite actions (critical for SignNow)
    "send": ["invite", "share"],
    "send document": ["invite to sign", "request signature"],
    "invite": ["send", "request"],
    "invite to sign": ["send document", "request signature"],
    "share": ["send", "distribute"],

    # CRUD operations
    "get": ["fetch", "retrieve", "read", "obtain", "list", "view"],
    "delete": ["remove", "cancel", "revoke"],
    "update": ["edit", "modify", "change", "patch"],
    "create": ["add", "new", "make", "generate"],
    "find": ["search", "lookup", "query", "locate"],
    "list": ["get all", "fetch", "retrieve"],
    "view": ["get", "read", "show", "display"],

    # Document actions
    "download": ["export", "save"],
    "upload": ["import", "add file"],
    "merge": ["combine", "join"],
    "copy": ["duplicate", "clone"],

    # Signing workflow
    "sign": ["signature", "execute"],
    "signature": ["sign", "signing"],
    "approve": ["accept", "confirm"],
    "reject": ["decline", "deny"],
    "cancel": ["revoke", "void", "delete"],

    # Communication
    "reminder": ["resend", "follow up", "notify", "nudge"],
    "resend": ["reminder", "send again", "re-send", "follow up"],
    "notify": ["alert", "inform", "reminder"],
    "email": ["mail", "send email"],
    "sms": ["text", "text message"],

    # Templates
    "template": ["blueprint", "preset"],
    "prefill": ["pre-fill", "populate", "autofill"],

    # Groups
    "group": ["bundle", "batch", "collection"],
    "folder": ["directory", "container"],
}


class SynonymExpander:
    """Expands text with synonyms for better search coverage."""

    def __init__(self, synonyms: dict[str, list[str]] | None = None):
        """Initialize with custom or default synonyms.

        Args:
            synonyms: Custom synonym dictionary. Uses defaults if None.
        """
        self.synonyms = synonyms or DEFAULT_SYNONYMS.copy()

    def add_synonym(self, word: str, synonyms: list[str]) -> None:
        """Add or update synonyms for a word."""
        self.synonyms[word.lower()] = synonyms

    def remove_synonym(self, word: str) -> None:
        """Remove a word from synonyms."""
        self.synonyms.pop(word.lower(), None)

    def enrich_text(self, text: str) -> str:
        """Enrich text with synonyms for indexing.

        Args:
            text: Original text to enrich.

        Returns:
            Text with synonyms appended.
        """
        text_lower = text.lower()
        additions: set[str] = set()

        for word, syns in self.synonyms.items():
            if word in text_lower:
                additions.update(syns)

        if additions:
            return f"{text} {' '.join(additions)}"

        return text

    def expand_query(self, query: str, max_expansions: int = 3) -> list[str]:
        """Expand query into multiple variants using synonyms.

        Args:
            query: Original search query.
            max_expansions: Maximum synonym expansions per word.

        Returns:
            List of query variants including original.
        """
        expansions = [query]
        query_lower = query.lower()

        for word, syns in self.synonyms.items():
            if word in query_lower:
                for syn in syns[:max_expansions]:
                    expanded = query_lower.replace(word, syn)
                    if expanded not in expansions:
                        expansions.append(expanded)

        return expansions


# Module-level default instance
_default_expander: SynonymExpander | None = None


def get_default_expander() -> SynonymExpander:
    """Get or create the default synonym expander."""
    global _default_expander
    if _default_expander is None:
        _default_expander = SynonymExpander()
    return _default_expander


def enrich_text_with_synonyms(text: str) -> str:
    """Enrich text with synonyms using default expander."""
    return get_default_expander().enrich_text(text)


def get_query_expansions(query: str) -> list[str]:
    """Get query expansions using default expander."""
    return get_default_expander().expand_query(query)
